/** @file   mine.h
 * @brief   Declaration of Mine - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_MINE_H
#define H_WWW_MINE_H

#include "gameobject.h"

namespace WeWantWar {

/** @class  Mine
 * @brief   Represents a mine that triggers/explodes when activated.
 * @author  Tomi Lamminsaari
 *
 */
class Mine : public GameObject
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** Datatype for minetypes */
  typedef int MineType;
  
  /** ID of anti-infantry mine */
  static const MineType ANTI_INFANTRY = 0;

  /** Index of the counter we use when timing the detonating delay */
  static const int DETONATING_COUNTER_INDEX = 1;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     minetype          Type of mine we should construct.
   */
	Mine( MineType minetype );


	/** Destructor
   */
	virtual ~Mine();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Mine
   */
  Mine( const Mine& rO );

	/** Assignment operator
   * @param     rO                Reference to another Mine
   * @return    Reference to us.
   */
  Mine& operator = ( const Mine& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this mine
   */
  virtual void update();
  
  /** Redraws this mine
   * @param     pQueue            Pointer to queue where the graphics
   *                              should be pushed to wait for being drawn.
   */
  virtual void redraw( RedrawQueue* pQueue );
  
  /** Makes sound
   * @param     id                ID of the sound we should make.
   */
  virtual void makeSound( GameObject::SoundID id ) const;
  
  /** This mine was hit by given bullet
   * @param     pB                Pointer to bullet that hit us.
   * @return    <code>false</code> since mines do not interact with bullets.
   */
  virtual bool hitByBullet( Bullet* pB );
  
  /** Kills this object
   */
  virtual void kill();


  ///
  /// Getter methods
  /// ==============
  
  /** Returns the type of this GameObject
   * @return    ObjectID::TYPE_MINE
   */
  virtual ObjectID::Type objectType() const;
  
  /** Tells if we're reloading.
   * @return    <code>false</code> but this value has no further meaning
   *            since Mines don't shoot.
   */
  virtual bool reloading() const;
  
  /** Returns the type of this mine.
   * @return    Type of this mine such as ANTI_INFANTRY
   */
  MineType mineType() const;
  


protected:

  /** Initializes the Anti-infantry mine
   */
  void initInfantryMine();
  
  /** Detonates and explodes this mine.
   */
  void detonateMine();
  

  ///
  /// Members
  /// =======

  /** Type of this mine */
  MineType    m_minetype;

  /** How much damage the exploding mine fragments cause. */
  int         m_fragmentDamage;
  
  /** The velocity of the mine fragments */
  float       m_fragmentVelocity;
  
  /** Number of fragments we emit when exploding */
  int         m_fragmentCount;
  
  /** How long we stay in detonating state */
  int         m_detonatingDelay;

  /** How far the fragments should fly */
  float       m_fragmentRange;

private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: mine.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:04+02  lamminsa
 * Initial revision
 *
 */
 
